<?php namespace App\Controllers\Api\Customer\Payment\PayPal;

use App\Controllers\PrivateController;
use App\Libraries\Settings;
use App\Models\AppsModel;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\SubscribesModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class CreatePaymentRequest extends PrivateController
{
    protected $paypalUrl = 'https://api.paypal.com/v2/checkout/orders';

    public function index(): ResponseInterface
    {
        if (!$this->validate($this->create_validation_type())) {
            return $this->respond(["message" => $this->validator->getErrors()], 400);
        }

        $uid = esc($this->request->getGet("uid"));

        $db = \Config\Database::connect();
        $db->transStart();

        try {
            $app = $this->getApp($uid);
            $this->checkActiveSubscription($app["id"]);
            $plan = $this->getPlan();
            $method = $this->getPayPalMethod();

            $order = $this->createPayPalOrder($method, $plan, $app["id"]);

            if (isset($order['links'][1]['href'])) {
                $db->transComplete();
                return $this->respond(["url" => $order['links'][1]['href']], 200);
            } else {
                throw new DatabaseException("PayPal order creation failed");
            }
        } catch (\Exception $e) {
            $db->transRollback();
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function getApp($uid)
    {
        $projects = new AppsModel();
        $app = $projects->where([
            "uid" => $uid,
            "user" => $this->userId,
            "deleted_at" => 0
        ])->select("id")->first();

        if (!$app) {
            throw new DatabaseException(lang("Message.message_14"));
        }

        return $app;
    }

    private function checkActiveSubscription($appId)
    {
        $subscribes = new SubscribesModel();
        $is_active = $subscribes->where([
            "app_id" => $appId,
            "expires_at >" => time(),
            "is_disable" => 0
        ])->countAllResults();

        if ($is_active) {
            throw new DatabaseException(lang("Message.message_86"));
        }
    }

    private function getPlan()
    {
        $plans = new PlansModel();
        $plan = $plans->where([
            "id" => (int)$this->request->getJsonVar("plan_id"),
            "status" => 1,
            "deleted_at" => 0
        ])->first();

        if (!$plan) {
            throw new DatabaseException(lang("Message.message_83"));
        }

        return $plan;
    }

    private function getPayPalMethod()
    {
        $methods = new DepositMethodsModel();
        $method = $methods->where([
            "name" => "PayPal",
            "status" => 1
        ])->first();

        if (!$method) {
            throw new DatabaseException(lang("Message.message_84"));
        }

        return $method;
    }

    private function createPayPalOrder($method, $plan, $appId)
    {
        $settings = new Settings();
        $frontUrl = rtrim($settings->get_config("site_url"), '/') . '/';

        $returnUrl = base_url('public/ipn/paypal/capture');
        $cancelUrl = $frontUrl . 'private/profile/subscribe';

        $clientId = $method["api_value_1"];
        $clientSecret = $method["api_value_2"];
        $auth = base64_encode("$clientId:$clientSecret");

        // Lấy mã tiền tệ từ bảng settings
        $currencyCode = $settings->get_config("currency_code");

        log_message('debug', 'PayPal Auth: ' . $auth);
        log_message('debug', 'Currency Code: ' . $currencyCode);

        $client = service('curlrequest');
        $response = $client->post($this->paypalUrl, [
            'headers' => [
                'Authorization' => "Basic $auth",
                'Content-Type' => 'application/json'
            ],
            'json' => [
                'intent' => 'CAPTURE',
                'purchase_units' => [[
                    'amount' => [
                        'currency_code' => $currencyCode,
                        'value' => $plan["price"]
                    ],
                    'custom_id' => $plan["id"]."_" .  $this->userId."_" . $appId,
                ]],
                'application_context' => [
                    'return_url' => $returnUrl,
                    'cancel_url' => $cancelUrl
                ]
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    private function create_validation_type(): array
    {
        return [
            "plan_id" => [
                "label" => lang("Fields.field_148"),
                "rules" => "required|numeric"
            ],
        ];
    }
}